# Configure PowerCLI to ignore invalid SSL certificates
Set-PowerCLIConfiguration -Scope User -InvalidCertificateAction Ignore -Confirm:$false

# Connect to the vCenter Server
Connect-VIServer -Server <vCenter_Server_IP> -User <username> -Password <password>

# Get all VMs
$allVMs = Get-VM

# Create a custom object to store snapshot details
$snapshotDetails = @()

# Loop through each VM to check for snapshots
foreach ($vm in $allVMs) {
    $snapshots = Get-Snapshot -VM $vm -ErrorAction SilentlyContinue

    if ($snapshots) {
        foreach ($snapshot in $snapshots) {
            $snapshotDetails += New-Object PSObject -Property @{
                VMName        = $vm.Name
                SnapshotName  = $snapshot.Name
                Created       = $snapshot.Created
                SizeMB        = [math]::round($snapshot.SizeMB, 2)
                AgeDays       = ((Get-Date) - $snapshot.Created).Days
            }
        }
    }
}

# Disconnect from the vCenter Server
Disconnect-VIServer -Server <vCenter_Server_IP> -Confirm:$false

# Format the snapshot details as an HTML table for the email body
$emailBody = @"
<html>
<head>
<style>
    table { font-family: Arial, sans-serif; border-collapse: collapse; width: 100%; }
    th, td { border: 1px solid #dddddd; text-align: left; padding: 8px; }
    tr:nth-child(even) { background-color: #f2f2f2; }
</style>
</head>
<body>
<h2>VMware Snapshot Report</h2>
<table>
    <tr>
        <th>VM Name</th>
        <th>Snapshot Name</th>
        <th>Created</th>
        <th>Size (MB)</th>
        <th>Age (Days)</th>
    </tr>
"@

$snapshotDetails | Sort-Object AgeDays -Descending | ForEach-Object {
    $emailBody += @"
    <tr>
        <td>$($_.VMName)</td>
        <td>$($_.SnapshotName)</td>
        <td>$($_.Created)</td>
        <td>$($_.SizeMB)</td>
        <td>$($_.AgeDays)</td>
    </tr>
"@
}

$emailBody += @"
</table>
</body>
</html>
"@

# Email parameters
$smtpServer = "<SMTP_Server>"
$smtpFrom = "<your_email@example.com>"
$smtpTo = "<recipient_email@example.com>"
$subject = "VMware Snapshot Report"
$body = $emailBody

# Send email with HTML body
Send-MailMessage -From $smtpFrom -To $smtpTo -Subject $subject -Body $body -SmtpServer $smtpServer -BodyAsHtml -Encoding UTF8

Write-Host "Email sent with snapshot details."
